"IAL"

************************************************************************
* Inverse assembler for the Western Design Center W65C02S
*
* INPUT_STATUS bits:
*             _
*   7       R/W
*           ___
*   6       RST
*           ___
*   5       NMI
*           ___
*   4       INT
*
*   3       SYNC
*
* Logic states must be clocked in on the falling edge of the 6502's
* PHI2o (phase-2-out) clock.
*
************************************************************************

    SEARCH_LIMIT    5
    DEFAULT_WIDTH   32
    MAPPED_WIDTH    32
    LABEL_TITLE     "       6502 mnemonic"
    BASE_TITLE      "            Hex"

************************************************************************
* Variables
************************************************************************

NEW_ADDRESS     VARIABLE    0   * address to fetch
LOW_BYTE        VARIABLE    0   * low byte of a 16-bit address
HIGH_BYTE       VARIABLE    0   * high byte of a 16-bit address
ADDRESS         VARIABLE    0   * 16-bit address
BYTE            VARIABLE    0   * 8-bit data byte
RELTEMP			VARIABLE	0	* 8-bit temporary storage (for REL_ADDR)
RELTEMP2		VARIABLE	0	* 8-bit temporary storage (for REL_ADDR)

************************************************************************
* Constants
************************************************************************

*** Strings

UNKNOWN         ASCII "unknown"
RESET           ASCII "reset"
MEM_WRITE       ASCII "memory write"
MEM_READ        ASCII "memory read"
INTERRUPT       ASCII "interrupt"
NONMASKABLE     ASCII "non-maskable interrupt"
* OPERAND         ASCII "operand"
OPERAND			ASCII ""

*** Tags

TAG_OPERAND     CONSTANT    1
TAG_NONE        CONSTANT    0

*** Output formats

DEC3_FMT		FORMAT 3,DEC,1       ; 3 bits, in decimal, 1 digit.
HEX7_FMT        FORMAT 7,HEX,2       ; 7 bits, in hex, 2 digits
HEX8_FMT        FORMAT 8,HEX,2       ; 8 bits, in hex, 2 digits
HEX16_FMT       FORMAT 16,HEX,4      ; 16 bits, in hex, 4 digits

************************************************************************
* Initialisation
************************************************************************

    SET         RETURN_FLAGS,0

    LOAD        INITIAL_ADDRESS
    STORE       NEW_ADDRESS

************************************************************************
* Start of inverse assembler code
************************************************************************

* output the data
    OUTPUT      "  "
    LOAD        INPUT_DATA           ; get the opcode
    OUTPUT      ACCUMULATOR,HEX8_FMT ; display as two hex characters
    OUTPUT      "   "


* is this byte tagged?
    LOAD        INPUT_TAG
    IF          0,0 = 0 THEN GOTO NOT_TAGGED

* yes, tell the user it's an operand then return
    OUTPUT      OPERAND
    RETURN

NOT_TAGGED
* what state is the CPU in?
    LOAD        INPUT_STATUS
    IF          0,0 = 1 THEN GOTO DECODE_OPCODE          ; opcode fetch

    CASE_OF     4,1
        OUTPUT  RESET                ; 0000  RESET
        OUTPUT  RESET                ; 0001  RESET
        OUTPUT  RESET                ; 0010  RESET
        OUTPUT  RESET                ; 0011  RESET
        OUTPUT  UNKNOWN              ; 0100  NMI AND INT (ILLEGAL)
        OUTPUT  NONMASKABLE          ; 0101  NMI
        OUTPUT  INTERRUPT            ; 0110  INT
        OUTPUT  MEM_WRITE            ; 0111  MEM WRITE
        OUTPUT  RESET                ; 1000  RESET
        OUTPUT  RESET                ; 1001  RESET
        OUTPUT  RESET                ; 1010  RESET
        OUTPUT  RESET                ; 1011  RESET
        OUTPUT  UNKNOWN              ; 1100  NMI AND INT (ILLEGAL)
        OUTPUT  NONMASKABLE          ; 1101  NMI
        OUTPUT  INTERRUPT            ; 1110  INT
        OUTPUT  MEM_READ             ; 1111  MEM READ
    CASE_END

    RETURN                           ; not an op fetch so exit

*** decode an opcode
DECODE_OPCODE
    LOAD        INPUT_DATA           ; get the opcode
    
* overrides
    IF 7,0 = 0CBH THEN GOTO WAI_IMP  ; WAI
    IF 7,0 = 0DBH THEN GOTO STP_IMP  ; STP

* opcode decoding table: decodes the LSB into an instruction group ID number
    CASE_OF     3,0                  ; LSB
        GOTO    GROUP_0              ; 0
        GOTO    GROUP_1              ; 1
        GOTO    GROUP_2              ; 2
        GOTO    BAD_OPCODE           ; 3
        GOTO    GROUP_4              ; 4
        GOTO    GROUP_5              ; 5
        GOTO    GROUP_6              ; 6
        GOTO    GROUP_7              ; 7
        GOTO    GROUP_8              ; 8
        GOTO    GROUP_9              ; 9
        GOTO    GROUP_A              ; A
        GOTO    BAD_OPCODE           ; B
        GOTO    GROUP_C              ; C
        GOTO    GROUP_D              ; D
        GOTO    GROUP_E              ; E
        GOTO    GROUP_F              ; F
    CASE_END

**************
* Group tables
**************

GROUP_0
    CASE_OF   7,4                    ; MSB
        GOTO  BRK_IMP
        GOTO  BPL_R
        GOTO  JSR_ABS
        GOTO  BMI_R
        GOTO  RTI_IMP
        GOTO  BVC_R
        GOTO  RTS_IMP
        GOTO  BVS_R
        GOTO  BRA_R
        GOTO  BCC_R
        GOTO  LDY_IMM
        GOTO  BCS_R
        GOTO  CPY_IMM
        GOTO  BNE_R
        GOTO  CPX_IMM
        GOTO  BEQ_R
    CASE_END
    RETURN

GROUP_1
    CASE_OF   7,4                    ; MSB
        GOTO  ORA_ZPII
        GOTO  ORA_ZPIIY
        GOTO  AND_ZPII
        GOTO  AND_ZPIIY
        GOTO  EOR_ZPII
        GOTO  EOR_ZPIIY
        GOTO  ADC_ZPII
        GOTO  ADC_ZPIIY
        GOTO  STA_ZPII
        GOTO  STA_ZPIIY
        GOTO  LDA_ZPII
        GOTO  LDA_ZPIIY
        GOTO  CMP_ZPII
        GOTO  CMP_ZPIIY
        GOTO  SBC_ZPII
        GOTO  SBC_ZPIIY
    CASE_END
    RETURN

GROUP_2
    CASE_OF   7,4                    ; MSB
        GOTO  BAD_OPCODE
        GOTO  ORA_ZPI
        GOTO  BAD_OPCODE
        GOTO  AND_ZPI
        GOTO  BAD_OPCODE
        GOTO  EOR_ZPI
        GOTO  BAD_OPCODE
        GOTO  ADC_ZPI
        GOTO  BAD_OPCODE
        GOTO  STA_ZPI
        GOTO  LDX_IMM
        GOTO  LDA_ZPI
        GOTO  BAD_OPCODE
        GOTO  CMP_ZPI
        GOTO  BAD_OPCODE
        GOTO  SBC_ZPI
    CASE_END
    RETURN

GROUP_4
    CASE_OF   7,4                    ; MSB
        GOTO  TSB_ZP
        GOTO  TRB_ZP
        GOTO  BIT_ZP
        GOTO  BIT_ZPIX
        GOTO  BAD_OPCODE
        GOTO  BAD_OPCODE
        GOTO  STZ_ZP
        GOTO  STZ_ZPIX
        GOTO  STY_ZP
        GOTO  STY_ZPIX
        GOTO  LDY_ZP
        GOTO  LDY_ZPIX
        GOTO  CPY_ZP
        GOTO  BAD_OPCODE
        GOTO  CPX_ZP
        GOTO  BAD_OPCODE
    CASE_END
    RETURN

GROUP_5
    CASE_OF   7,4                    ; MSB
        GOTO  ORA_ZP
        GOTO  ORA_ZPIX
        GOTO  AND_ZP
        GOTO  AND_ZPIX
        GOTO  EOR_ZP
        GOTO  EOR_ZPIX
        GOTO  ADC_ZP
        GOTO  ADC_ZPIX
        GOTO  STA_ZP
        GOTO  STA_ZPIX
        GOTO  LDA_ZP
        GOTO  LDA_ZPIX
        GOTO  CMP_ZP
        GOTO  CMP_ZPIX
        GOTO  SBC_ZP
        GOTO  SBC_ZPIX
    CASE_END
    RETURN

GROUP_6
    CASE_OF   7,4                    ; MSB
        GOTO  ASL_ZP
        GOTO  ASL_ZPIX
        GOTO  ROL_ZP
        GOTO  ROL_ZPIX
        GOTO  LSR_ZP
        GOTO  LSR_ZPIX
        GOTO  ROR_ZP
        GOTO  ROR_ZPIX
        GOTO  STX_ZP
        GOTO  STX_ZPIX
        GOTO  LDX_ZP
        GOTO  LDX_ZPIX
        GOTO  DEC_ZP
        GOTO  DEC_ZPIX
        GOTO  INC_ZP
        GOTO  INC_ZPIX
    CASE_END
    RETURN

GROUP_7
    CASE_OF   7,4                    ; MSB
        GOTO  RMB_ZP
        GOTO  RMB_ZP
        GOTO  RMB_ZP
        GOTO  RMB_ZP
        GOTO  RMB_ZP
        GOTO  RMB_ZP
        GOTO  RMB_ZP
        GOTO  RMB_ZP
        GOTO  SMB_ZP
        GOTO  SMB_ZP
        GOTO  SMB_ZP
        GOTO  SMB_ZP
        GOTO  SMB_ZP
        GOTO  SMB_ZP
        GOTO  SMB_ZP
        GOTO  SMB_ZP
    CASE_END
    RETURN

GROUP_8
    CASE_OF   7,4                    ; MSB
        GOTO  PHP_IMP
        GOTO  CLC_IMP
        GOTO  PLP_IMP
        GOTO  SEC_IMP
        GOTO  PHA_IMP
        GOTO  CLI_IMP
        GOTO  PLA_IMP
        GOTO  SEI_IMP
        GOTO  DEY_IMP
        GOTO  TYA_IMP
        GOTO  TAY_IMP
        GOTO  CLV_IMP
        GOTO  INY_IMP
        GOTO  CLD_IMP
        GOTO  INX_IMP
        GOTO  SED_IMP
    CASE_END
    RETURN

GROUP_9
    CASE_OF   7,4                    ; MSB
        GOTO  ORA_IMM
        GOTO  ORA_AIY
        GOTO  AND_IMM
        GOTO  AND_AIY
        GOTO  EOR_IMM
        GOTO  EOR_AIY
        GOTO  ADC_IMM
        GOTO  ADC_AIY
        GOTO  BIT_IMM
        GOTO  STA_AIY
        GOTO  LDA_IMM
        GOTO  LDA_AIY
        GOTO  CMP_IMM
        GOTO  CMP_AIY
        GOTO  SBC_IMM
        GOTO  SBC_AIY
    CASE_END
    RETURN

GROUP_A
    CASE_OF   7,4                    ; MSB
        GOTO  ASL_A
        GOTO  INC_A
        GOTO  ROL_A
        GOTO  DEC_A
        GOTO  LSR_A
        GOTO  PHY_IMP
        GOTO  ROR_A
        GOTO  PLY_IMP
        GOTO  TXA_IMP
        GOTO  TXS_IMP
        GOTO  TAX_IMP
        GOTO  TSX_IMP
        GOTO  DEX_IMP
        GOTO  PHX_IMP
        GOTO  NOP_IMP
        GOTO  PLX_IMP
    CASE_END
    RETURN

GROUP_C
    CASE_OF   7,4                    ; MSB
        GOTO  TSB_ABS
        GOTO  TRB_ABS
        GOTO  BIT_ABS
        GOTO  BIT_AIX
        GOTO  JMP_ABS
        GOTO  BAD_OPCODE
        GOTO  JMP_AI
        GOTO  JMP_AII
        GOTO  STY_ABS
        GOTO  STZ_ABS
        GOTO  LDY_ABS
        GOTO  LDY_AIX
        GOTO  CPY_ABS
        GOTO  BAD_OPCODE
        GOTO  CPX_ABS
        GOTO  BAD_OPCODE
    CASE_END
    RETURN

GROUP_D
    CASE_OF   7,4                    ; MSB
        GOTO  ORA_ABS
        GOTO  ORA_AIX
        GOTO  AND_ABS
        GOTO  AND_AIX
        GOTO  EOR_ABS
        GOTO  EOR_AIX
        GOTO  ADC_ABS
        GOTO  ADC_AIX
        GOTO  STA_ABS
        GOTO  STA_AIX
        GOTO  LDA_ABS
        GOTO  LDA_AIX
        GOTO  CMP_ABS
        GOTO  CMP_AIX
        GOTO  SBC_ABS
        GOTO  SBC_AIX
    CASE_END
    RETURN

GROUP_E
    CASE_OF   7,4                    ; MSB
        GOTO  ASL_ABS
        GOTO  ASL_AIX
        GOTO  ROL_ABS
        GOTO  ROL_AIX
        GOTO  LSR_ABS
        GOTO  LSR_AIX
        GOTO  ROR_ABS
        GOTO  ROR_AIX
        GOTO  STX_ABS
        GOTO  STZ_AIX
        GOTO  LDX_ABS
        GOTO  LDX_AIY
        GOTO  DEC_ABS
        GOTO  DEC_AIX
        GOTO  INC_ABS
        GOTO  INC_AIX
    CASE_END
    RETURN

GROUP_F
    CASE_OF   7,4                    ; MSB
        GOTO  BBR_RELZP
        GOTO  BBR_RELZP
        GOTO  BBR_RELZP
        GOTO  BBR_RELZP
        GOTO  BBR_RELZP
        GOTO  BBR_RELZP
        GOTO  BBR_RELZP
        GOTO  BBR_RELZP
        GOTO  BBS_RELZP
        GOTO  BBS_RELZP
        GOTO  BBS_RELZP
        GOTO  BBS_RELZP
        GOTO  BBS_RELZP
        GOTO  BBS_RELZP
        GOTO  BBS_RELZP
        GOTO  BBS_RELZP
    CASE_END
    RETURN


************************************************************************
* Single-byte instructions
************************************************************************

*******
** group 0 single-byte
*******
BRK_IMP
    OUTPUT      "BRK"
    RETURN
RTI_IMP
    OUTPUT      "RTI"
    RETURN
RTS_IMP
    OUTPUT      "RTS"
    RETURN

*******
** group 8 single-byte
*******
PHP_IMP
    OUTPUT      "PHP"
    RETURN
CLC_IMP
    OUTPUT      "CLC"
    RETURN
PLP_IMP
    OUTPUT      "PLP"
    RETURN
SEC_IMP
    OUTPUT      "SEC"
    RETURN
PHA_IMP
    OUTPUT      "PHA"
    RETURN
CLI_IMP
    OUTPUT      "CLI"
    RETURN
PLA_IMP
    OUTPUT      "PLA"
    RETURN
SEI_IMP
    OUTPUT      "SEI"
    RETURN
DEY_IMP
    OUTPUT      "DEY"
    RETURN
TYA_IMP
    OUTPUT      "TYA"
    RETURN
TAY_IMP
    OUTPUT      "TAY"
    RETURN
CLV_IMP
    OUTPUT      "CLV"
    RETURN
INY_IMP
    OUTPUT      "INY"
    RETURN
CLD_IMP
    OUTPUT      "CLD"
    RETURN
INX_IMP
    OUTPUT      "INX"
    RETURN
SED_IMP
    OUTPUT      "SED"
    RETURN

*******
** group A single-byte
*******
ASL_A
    OUTPUT      "ASL  A"
    RETURN
INC_A
    OUTPUT      "INC  A"
    RETURN
ROL_A
    OUTPUT      "ROL  A"
    RETURN
DEC_A
    OUTPUT      "DEC  A"
    RETURN
LSR_A
    OUTPUT      "LSR  A"
    RETURN
ROR_A
    OUTPUT      "ROR  A"
    RETURN
PHY_IMP
    OUTPUT      "PHY"
    RETURN
PLY_IMP
    OUTPUT      "PLY"
    RETURN
TXA_IMP
    OUTPUT      "TXA"
    RETURN
TXS_IMP
    OUTPUT      "TXS"
    RETURN
TAX_IMP
    OUTPUT      "TAX"
    RETURN
TSX_IMP
    OUTPUT      "TSX"
    RETURN
DEX_IMP
    OUTPUT      "DEX"
    RETURN
PHX_IMP
    OUTPUT      "PHX"
    RETURN
NOP_IMP
    OUTPUT      "NOP"
    RETURN
PLX_IMP
    OUTPUT      "PLX"
    RETURN

*******
** group B single-byte
*******
WAI_IMP
    OUTPUT      "WAI"
    RETURN
STP_IMP
    OUTPUT      "STP"
    RETURN

************************************************************************
* Double-byte instructions
************************************************************************

*******
** group 0 double-byte
*******
BPL_R
    OUTPUT      "BPL "
    GOTO        OUTPUT_REL_ADDR
BMI_R
    OUTPUT      "BMI "
    GOTO        OUTPUT_REL_ADDR
BVC_R
    OUTPUT      "BVC "
    GOTO        OUTPUT_REL_ADDR
BVS_R
    OUTPUT      "BVS "
    GOTO        OUTPUT_REL_ADDR
BRA_R
    OUTPUT      "BRA "
    GOTO        OUTPUT_REL_ADDR
BCC_R
    OUTPUT      "BCC "
    GOTO        OUTPUT_REL_ADDR
LDY_IMM
    OUTPUT      "LDY "
    GOTO        OUTPUT_IMM
BCS_R
    OUTPUT      "BCS "
    GOTO        OUTPUT_REL_ADDR
CPY_IMM
    OUTPUT      "CPY "
    GOTO        OUTPUT_IMM
BNE_R
    OUTPUT      "BNE "
    GOTO        OUTPUT_REL_ADDR
CPX_IMM
    OUTPUT      "CPX "
    GOTO        OUTPUT_IMM
BEQ_R
    OUTPUT      "BEQ "
    GOTO        OUTPUT_REL_ADDR

*******
** group 1 double-byte
*******
ORA_ZPII
    OUTPUT      "ORA "
    GOTO        OUTPUT_ZPII
ORA_ZPIIY
    OUTPUT      "ORA "
    GOTO        OUTPUT_ZPIIY
AND_ZPII
    OUTPUT      "AND "
    GOTO        OUTPUT_ZPII
AND_ZPIIY
    OUTPUT      "AND "
    GOTO        OUTPUT_ZPIIY
EOR_ZPII
    OUTPUT      "EOR "
    GOTO        OUTPUT_ZPII
EOR_ZPIIY
    OUTPUT      "EOR "
    GOTO        OUTPUT_ZPIIY
ADC_ZPII
    OUTPUT      "ADC "
    GOTO        OUTPUT_ZPII
ADC_ZPIIY
    OUTPUT      "ADC "
    GOTO        OUTPUT_ZPIIY
STA_ZPII
    OUTPUT      "STA "
    GOTO        OUTPUT_ZPII
STA_ZPIIY
    OUTPUT      "STA "
    GOTO        OUTPUT_ZPIIY
LDA_ZPII
    OUTPUT      "LDA "
    GOTO        OUTPUT_ZPII
LDA_ZPIIY
    OUTPUT      "LDA "
    GOTO        OUTPUT_ZPIIY
CMP_ZPII
    OUTPUT      "CMP "
    GOTO        OUTPUT_ZPII
CMP_ZPIIY
    OUTPUT      "CMP "
    GOTO        OUTPUT_ZPIIY
SBC_ZPII
    OUTPUT      "SBC "
    GOTO        OUTPUT_ZPII
SBC_ZPIIY
    OUTPUT      "SBC "
    GOTO        OUTPUT_ZPIIY

*******
** group 2 double-byte
*******         
ORA_ZPI
    OUTPUT      "ORA "
    GOTO        OUTPUT_ZPI
AND_ZPI
    OUTPUT      "AND "
    GOTO        OUTPUT_ZPI
EOR_ZPI
    OUTPUT      "EOR "
    GOTO        OUTPUT_ZPI
ADC_ZPI
    OUTPUT      "ADC "
    GOTO        OUTPUT_ZPI
STA_ZPI
    OUTPUT      "STA "
    GOTO        OUTPUT_ZPI
LDX_IMM
    OUTPUT      "LDX "
    GOTO        OUTPUT_IMM
LDA_ZPI
    OUTPUT      "LDA "
    GOTO        OUTPUT_ZPI
CMP_ZPI
    OUTPUT      "CMP "
    GOTO        OUTPUT_ZPI
SBC_ZPI
    OUTPUT      "SBC "
    GOTO        OUTPUT_ZPI

*******
** group 4 double-byte
*******
TSB_ZP
	OUTPUT		"TSB "
    GOTO        OUTPUT_ZP
TRB_ZP
	OUTPUT		"TRB "
    GOTO        OUTPUT_ZP
BIT_ZP
	OUTPUT		"BIT "
    GOTO        OUTPUT_ZP
BIT_ZPIX
	OUTPUT		"BIT "
    GOTO        OUTPUT_ZPIX
STZ_ZP
	OUTPUT		"STZ "
    GOTO        OUTPUT_ZP
STZ_ZPIX
	OUTPUT		"STZ "
    GOTO        OUTPUT_ZPIX
STY_ZP
	OUTPUT		"STY "
    GOTO        OUTPUT_ZP
STY_ZPIX
	OUTPUT		"STY "
    GOTO        OUTPUT_ZPIX
LDY_ZP
	OUTPUT		"LDY "
    GOTO        OUTPUT_ZP
LDY_ZPIX
	OUTPUT		"LDY "
    GOTO        OUTPUT_ZPIX
CPY_ZP
	OUTPUT		"CPY "
	GOTO		OUTPUT_ZP
CPX_ZP
	OUTPUT		"CPX "
	GOTO		OUTPUT_ZP

*******
** group 5 double-byte
*******
ORA_ZP
	OUTPUT		"ORA "
	GOTO		OUTPUT_ZP
ORA_ZPIX
	OUTPUT		"ORA "
	GOTO		OUTPUT_ZPIX
AND_ZP
	OUTPUT		"AND "
	GOTO		OUTPUT_ZP
AND_ZPIX
	OUTPUT		"AND "
	GOTO		OUTPUT_ZPIX
EOR_ZP
	OUTPUT		"EOR "
	GOTO		OUTPUT_ZP
EOR_ZPIX
	OUTPUT		"EOR "
	GOTO		OUTPUT_ZPIX
ADC_ZP
	OUTPUT		"ADC "
	GOTO		OUTPUT_ZP
ADC_ZPIX
	OUTPUT		"ADC "
	GOTO		OUTPUT_ZPIX
STA_ZP
	OUTPUT		"STA "
	GOTO		OUTPUT_ZP
STA_ZPIX
	OUTPUT		"STA "
	GOTO		OUTPUT_ZPIX
LDA_ZP
	OUTPUT		"LDA "
	GOTO		OUTPUT_ZP
LDA_ZPIX
	OUTPUT		"LDA "
	GOTO		OUTPUT_ZPIX
CMP_ZP
	OUTPUT		"CMP "
	GOTO		OUTPUT_ZP
CMP_ZPIX
	OUTPUT		"CMP "
	GOTO		OUTPUT_ZPIX
SBC_ZP
	OUTPUT		"SBC "
	GOTO		OUTPUT_ZP
SBC_ZPIX
	OUTPUT		"SBC "
	GOTO		OUTPUT_ZPIX

*******
** group 6 double-byte
*******
ASL_ZP
	OUTPUT		"ASL "
	GOTO		OUTPUT_ZP
ASL_ZPIX
	OUTPUT		"ASL "
	GOTO		OUTPUT_ZPIX
ROL_ZP
	OUTPUT		"ROL "
	GOTO		OUTPUT_ZP
ROL_ZPIX
	OUTPUT		"ROL "
	GOTO		OUTPUT_ZPIX
LSR_ZP
	OUTPUT		"LSR "
	GOTO		OUTPUT_ZP
LSR_ZPIX
	OUTPUT		"LSR "
	GOTO		OUTPUT_ZPIX
ROR_ZP
	OUTPUT		"ROR "
	GOTO		OUTPUT_ZP
ROR_ZPIX
	OUTPUT		"ROR "
	GOTO		OUTPUT_ZPIX
STX_ZP
	OUTPUT		"STX "
	GOTO		OUTPUT_ZP
STX_ZPIX
	OUTPUT		"STX "
	GOTO		OUTPUT_ZPIX
LDX_ZP
	OUTPUT		"LDX "
	GOTO		OUTPUT_ZP
LDX_ZPIX
	OUTPUT		"LDX "
	GOTO		OUTPUT_ZPIX
DEC_ZP
	OUTPUT		"DEC "
	GOTO		OUTPUT_ZP
DEC_ZPIX
	OUTPUT		"DEC "
	GOTO		OUTPUT_ZPIX
INC_ZP
	OUTPUT		"INC "
	GOTO		OUTPUT_ZP
INC_ZPIX
	OUTPUT		"INC "
	GOTO		OUTPUT_ZPIX


*******
** group 7 double-byte
*******
RMB_ZP
	OUTPUT		"RMB"
	ROTATE		RIGHT,4
	OUTPUT		ACCUMULATOR,DEC3_FMT
	OUTPUT		" "
	GOTO		OUTPUT_ZP
SMB_ZP
	OUTPUT		"SMB"
	ROTATE		RIGHT,4
	OUTPUT		ACCUMULATOR,DEC3_FMT
	OUTPUT		" "
	GOTO		OUTPUT_ZP

*******
** group 9 double-byte
*******
ORA_IMM
	OUTPUT		"ORA "
	GOTO		OUTPUT_IMM
AND_IMM
	OUTPUT		"AND "
	GOTO		OUTPUT_IMM
EOR_IMM
	OUTPUT		"EOR "
	GOTO		OUTPUT_IMM
ADC_IMM
	OUTPUT		"ADC "
	GOTO		OUTPUT_IMM
BIT_IMM
	OUTPUT		"BIT "
	GOTO		OUTPUT_IMM
LDA_IMM
	OUTPUT		"LDA "
	GOTO		OUTPUT_IMM
CMP_IMM
	OUTPUT		"CMP "
	GOTO		OUTPUT_IMM
SBC_IMM
	OUTPUT		"SBC "
	GOTO		OUTPUT_IMM


************************************************************************
* Triple-byte instructions
************************************************************************

*******
** group 0 triple-byte
*******
JSR_ABS
    OUTPUT      "JSR "
    GOTO        OUTPUT_ABS

*******
** group 9 triple-byte
*******
ORA_AIY
	OUTPUT		"AND "
	GOTO		OUTPUT_AIY
AND_AIY
	OUTPUT		"AND "
	GOTO		OUTPUT_AIY
EOR_AIY
	OUTPUT		"EOR "
	GOTO		OUTPUT_AIY
ADC_AIY
	OUTPUT		"ADC "
	GOTO		OUTPUT_AIY
STA_AIY
	OUTPUT		"STA "
	GOTO		OUTPUT_AIY
LDA_AIY
	OUTPUT		"LDA "
	GOTO		OUTPUT_AIY
CMP_AIY
	OUTPUT		"CMP "
	GOTO		OUTPUT_AIY
SBC_AIY
	OUTPUT		"SBC "
	GOTO		OUTPUT_AIY

*******
** group C triple-byte
*******
TSB_ABS
	OUTPUT		"TSB "
	GOTO		OUTPUT_ABS
TRB_ABS
	OUTPUT		"TRB "
	GOTO		OUTPUT_ABS
BIT_ABS
	OUTPUT		"BIT "
	GOTO		OUTPUT_ABS
BIT_AIX
	OUTPUT		"BIT "
	GOTO		OUTPUT_AIX
JMP_ABS
	OUTPUT		"JMP "
	GOTO		OUTPUT_ABS
JMP_AI
	OUTPUT		"JMP "
	GOTO		OUTPUT_AI
JMP_AII
	OUTPUT		"JMP "
	GOTO		OUTPUT_AII
STY_ABS
	OUTPUT		"STY "
	GOTO		OUTPUT_ABS
STZ_ABS
	OUTPUT		"STZ "
	GOTO		OUTPUT_ABS
LDY_ABS
	OUTPUT		"LDY "
	GOTO		OUTPUT_ABS
LDY_AIX
	OUTPUT		"LDY "
	GOTO		OUTPUT_AIX
CPY_ABS
	OUTPUT		"CPY "
	GOTO		OUTPUT_ABS
CPX_ABS
	OUTPUT		"CPX "
	GOTO		OUTPUT_ABS

*******
** group D triple-byte
*******
ORA_ABS
	OUTPUT		"ORA "
	GOTO		OUTPUT_ABS
ORA_AIX
	OUTPUT		"ORA "
	GOTO		OUTPUT_AIX
AND_ABS
	OUTPUT		"AND "
	GOTO		OUTPUT_ABS
AND_AIX
	OUTPUT		"AND "
	GOTO		OUTPUT_AIX
EOR_ABS
	OUTPUT		"EOR "
	GOTO		OUTPUT_ABS
EOR_AIX
	OUTPUT		"EOR "
	GOTO		OUTPUT_AIX
ADC_ABS
	OUTPUT		"ADC "
	GOTO		OUTPUT_ABS
ADC_AIX
	OUTPUT		"ADC "
	GOTO		OUTPUT_AIX
STA_ABS
	OUTPUT		"STA "
	GOTO		OUTPUT_ABS
STA_AIX
	OUTPUT		"STA "
	GOTO		OUTPUT_AIX
LDA_ABS
	OUTPUT		"LDA "
	GOTO		OUTPUT_ABS
LDA_AIX
	OUTPUT		"LDA "
	GOTO		OUTPUT_AIX
CMP_ABS
	OUTPUT		"CMP "
	GOTO		OUTPUT_ABS
CMP_AIX
	OUTPUT		"CMP "
	GOTO		OUTPUT_AIX
SBC_ABS
	OUTPUT		"SBC "
	GOTO		OUTPUT_ABS
SBC_AIX
	OUTPUT		"SBC "
	GOTO		OUTPUT_AIX

*******
** group E triple-byte
*******
ASL_ABS
	OUTPUT		"ASL "
	GOTO		OUTPUT_ABS
ASL_AIX
	OUTPUT		"ASL "
	GOTO		OUTPUT_AIX
ROL_ABS
	OUTPUT		"ROL "
	GOTO		OUTPUT_ABS
ROL_AIX
	OUTPUT		"ROL "
	GOTO		OUTPUT_AIX
LSR_ABS
	OUTPUT		"LSR "
	GOTO		OUTPUT_ABS
LSR_AIX
	OUTPUT		"LSR "
	GOTO		OUTPUT_AIX
ROR_ABS
	OUTPUT		"ROR "
	GOTO		OUTPUT_ABS
ROR_AIX
	OUTPUT		"ROR "
	GOTO		OUTPUT_AIX
STX_ABS
	OUTPUT		"STX "
	GOTO		OUTPUT_ABS
STZ_AIX
	OUTPUT		"STZ "
	GOTO		OUTPUT_AIX
LDX_ABS
	OUTPUT		"LDX "
	GOTO		OUTPUT_ABS
LDX_AIY
	OUTPUT		"LDX "
	GOTO		OUTPUT_AIY
DEC_ABS
	OUTPUT		"DEC "
	GOTO		OUTPUT_ABS
DEC_AIX
	OUTPUT		"DEC "
	GOTO		OUTPUT_AIX
INC_ABS
	OUTPUT		"INC "
	GOTO		OUTPUT_ABS
INC_AIX
	OUTPUT		"INC "
	GOTO		OUTPUT_AIX

*******
** group F triple-byte
*******
BBR_RELZP
	OUTPUT		"BBR"
	ROTATE		RIGHT,4
	OUTPUT		ACCUMULATOR,DEC3_FMT
	OUTPUT		" "
	CALL		OUTPUT_ZP
	OUTPUT		", "
	GOTO		OUTPUT_REL_ADDR

BBS_RELZP
	OUTPUT		"BBS"
	ROTATE		RIGHT,4
	OUTPUT		ACCUMULATOR,DEC3_FMT
	OUTPUT		" "
	CALL		OUTPUT_ZP
	OUTPUT		", "
	GOTO		OUTPUT_REL_ADDR


************************************************************************
* Output routines
************************************************************************

*** Output a relative address
OUTPUT_REL_ADDR
    CALL        READ_OPERAND
    STORE		RELTEMP
    STORE		RELTEMP2

	IF 7,7 = 1	THEN GOTO RELA_NEGATIVE
RELA_POSITIVE
* relative addr is positive
	ADD			INPUT_ADDRESS
	ADD			01h
	ADD			RELTEMP
	GOTO		RELA_DONE
	AND			0FFFFh
RELA_NEGATIVE
* relative addr is negative
	TWOS_COMPLEMENT
	AND			07Fh
	STORE		RELTEMP
	LOAD		INPUT_ADDRESS
	SUBTRACT	RELTEMP
	ADD			01h
	AND			0FFFFh
RELA_DONE
	CALL		ADDR_MAP			* display either the address or a symbol
RELA_ADR
	LOAD	RELTEMP2
    IF 7,7 = 0  THEN OUTPUT "  (+$"
    IF 7,7 = 1  THEN OUTPUT "  (-$"
    IF 7,7 = 1  THEN TWOS_COMPLEMENT
    OUTPUT      ACCUMULATOR,HEX7_FMT
    OUTPUT		")"
    RETURN

*** Output an Absolute Indexed Indirect "(a, x)" operand
OUTPUT_AII
    CALL        READ_ADDRESS
    OUTPUT      "($"
    OUTPUT      ACCUMULATOR,HEX16_FMT
    OUTPUT      ", x)"
    RETURN

*** Output an Absolute Indexed With X "a, x" operand
OUTPUT_AIX
    CALL        READ_ADDRESS
    OUTPUT      "$"
    OUTPUT      ACCUMULATOR,HEX16_FMT
    OUTPUT      ", x"
    RETURN

*** Output an Absolute Indexed With Y "a, y" operand
OUTPUT_AIY
    CALL        READ_ADDRESS
    OUTPUT      "$"
    OUTPUT      ACCUMULATOR,HEX16_FMT
    OUTPUT      ", y"
    RETURN

*** Output an Absolute Indirect "(a)" operand
OUTPUT_AI
    CALL        READ_ADDRESS
    OUTPUT      "($"
    OUTPUT      ACCUMULATOR,HEX16_FMT
    OUTPUT      ")"
    RETURN

*** Output an Immediate "#$" operand
OUTPUT_IMM
    CALL        READ_OPERAND
    OUTPUT      "#$"
    OUTPUT      ACCUMULATOR,HEX8_FMT
    RETURN

*** Output a Zero Page "zp" operand
OUTPUT_ZP
    CALL        READ_OPERAND
    OUTPUT      "$"
    OUTPUT      ACCUMULATOR,HEX8_FMT
    RETURN

*** Output a Zero Page Indexed Indirect "(zp, x)" operand
OUTPUT_ZPII
    CALL        READ_OPERAND
    OUTPUT      "($"
    OUTPUT      ACCUMULATOR,HEX8_FMT
    OUTPUT      ", x)"
    RETURN

*** Output a Zero Page Indexed With X "zp, x" operand
OUTPUT_ZPIX
    CALL        READ_OPERAND
    OUTPUT      "$"
    OUTPUT      ACCUMULATOR,HEX8_FMT
    OUTPUT      ", x"
    RETURN

*** Output a Zero Page Indexed With Y "zp, y" operand
OUTPUT_ZPIY
    CALL        READ_OPERAND
    OUTPUT      "$"
    OUTPUT      ACCUMULATOR,HEX8_FMT
    OUTPUT      ", y"
    RETURN

*** Output a Zero Page Indirect "(zp)" operand
OUTPUT_ZPI
    CALL        READ_OPERAND
    OUTPUT      "($"
    OUTPUT      ACCUMULATOR,HEX8_FMT
    OUTPUT      ")"
    RETURN

*** Output a Zero Page Indexed With Y "(zp), y" operand
OUTPUT_ZPIIY
    CALL        READ_OPERAND
    OUTPUT      "($"
    OUTPUT      ACCUMULATOR,HEX8_FMT
    OUTPUT      "), y"
    RETURN

*** Output an Absolute "a" operand
OUTPUT_ABS
    CALL        READ_ADDRESS
    LOAD        ADDRESS
    CALL		ADDR_MAP
    RETURN
    
*** Output either a $-prefixed address or a symbol
ADDR_MAP
    IF_NOT_MAPPED THEN GOTO ADDR_MAP_NOT
	RETURN
ADDR_MAP_NOT
	OUTPUT	"$"
	OUTPUT	ACCUMULATOR,HEX16_FMT
	RETURN

************************************************************************
BAD_OPCODE
    OUTPUT      "Unrecognised opcode"
    RETURN
************************************************************************

************************************************************************
* Read an 8-bit operand
************************************************************************
READ_OPERAND
    LOAD        INPUT_ADDRESS        ; accumulator = current address
    STORE       NEW_ADDRESS          ; NEW_ADDRESS = accumulator

    INCREMENT   NEW_ADDRESS          ; find the addr of the LS byte
    INPUT       ABS,NEW_ADDRESS      ; get the byte
    LOAD        INPUT_DATA           ; into the accumulator
    STORE       BYTE                 ; and save it in BYTE

    TAG_WITH    TAG_OPERAND          ; tag it as an operand
    RETURN                           ; exit

************************************************************************
* Read a 16-bit address
************************************************************************
READ_ADDRESS
    LOAD        INPUT_ADDRESS        ; accumulator = current address
    STORE       NEW_ADDRESS          ; NEW_ADDRESS = accumulator

    INCREMENT   NEW_ADDRESS          ; find the addr of the LS byte
    INPUT       ABS,NEW_ADDRESS      ; get the LS byte of the destination
    LOAD        INPUT_DATA           ; address into the accumulator
    STORE       LOW_BYTE             ; and save it in LOW_BYTE
    TAG_WITH    TAG_OPERAND          ; tag it as an operand

    INCREMENT   NEW_ADDRESS          ; find the addr of the MS byte
    INPUT       ABS,NEW_ADDRESS      ; get the MS byte of the destination
    LOAD        INPUT_DATA           ; address into the accumulator
    STORE       HIGH_BYTE            ; and save it in HIGH_BYTE
    TAG_WITH    TAG_OPERAND          ; tag it as an operand

    ROTATE      LEFT,8               ; rotate left 8 bits
    INCLUSIVE_OR LOW_BYTE            ; OR with low byte
    AND         0FFFFH               ; only low 16 bits are valid
    STORE       ADDRESS              ; store address

    RETURN

